const express = require('express');
const router = express.Router();
const Waitlist = require('../models/Waitlist');
// UPDATED: Import both client and admin email functions
const { sendWaitlistClientEmail, sendWaitlistAdminNotification } = require('../utils/email'); 

// @route POST /api/waitlist
// @desc Register a new user to the waitlist
// @access Public
router.post('/', async (req, res) => {
    // UPDATED DESTRUCTURING: Includes phone and array of preferredContacts
    const { name, email, phone, preferredContacts = ['email'] } = req.body;

    // --- 1. Basic Validation ---
    if (!name || !email || !preferredContacts || preferredContacts.length === 0) {
        return res.status(400).json({ message: 'Name, email, and at least one preferred contact method are required.' });
    }

    // --- 2. Phone Validation (Required if call or message is selected) ---
    const requiresPhone = preferredContacts.includes('call') || preferredContacts.includes('message');
    if (requiresPhone && !phone) {
        return res.status(400).json({ message: 'Phone number is required for Call or WhatsApp/Text contact.' });
    }

    try {
        const existingUser = await Waitlist.findOne({ email });

        if (existingUser) {
            return res.status(400).json({ message: 'This email is already on the waitlist.' });
        }

        // --- 3. Save New Waitlist Entry ---
        const newUser = new Waitlist({ 
            name, 
            email, 
            phone, // Save phone number
            preferredContacts // Save array of contacts
        });
        await newUser.save();

        // --- 4. Send Emails ---
        // 4a. Send confirmation email to the client (fancy template)
        await sendWaitlistClientEmail(email, name);
        
        // 4b. Send notification to the Admin (plain text summary)
        await sendWaitlistAdminNotification(newUser); 

        res.status(201).json({ message: 'Successfully joined the waitlist!' });

    } catch (error) {
        console.error(error);
        // Handle MongoDB duplicate key error (11000) more gracefully
        if (error.code === 11000) {
            return res.status(400).json({ message: 'This email is already registered.' });
        }
        res.status(500).json({ message: 'Server error during registration.' });
    }
});

module.exports = router;