const express = require('express');
const router = express.Router();
const Waitlist = require('../models/Waitlist');
const Product = require('../models/Product'); // To fetch recommendations
const QuizSubmission = require('../models/QuizSubmission'); // Needed for submission record
const { sendQuizRecommendationEmail } = require('../utils/quizEmail'); 

// Helper to determine product recommendation based on quiz result (Mock)
const getRecommendations = (resultType) => {
    // NOTE: This should ultimately query a robust recommendation mapping system.
    if (resultType === 'fresh') return ['PUKU', 'LUEN'];
    if (resultType === 'dark') return ['LUEN', 'LOLITA'];
    if (resultType === 'floral') return ['PUKU', 'LAKEKHU'];
    if (resultType === 'warm') return ['LOLITA', 'LAKEKHU'];
    return [];
};

// Helper function to calculate the dominant scent type from quiz answers
const calculateScentType = (answers) => {
    // FIX: Use optional chaining and default to empty object to prevent TypeError
    const safeAnswers = answers || {}; 
    
    // Convert answers map values to an array, ignoring the first two steps 
    // (which are assumed to be Name and Buying For, based on frontend sequence)
    const scentAnswers = Object.values(safeAnswers).slice(2); 

    const counts = { fresh: 0, floral: 0, warm: 0, dark: 0 };
    scentAnswers.forEach(answer => {
        if (counts.hasOwnProperty(answer)) {
            counts[answer]++;
        }
    });

    // Determine the result based on the highest count, defaulting to 'dark'
    return Object.keys(counts).reduce((a, b) => counts[a] > counts[b] ? a : b, 'dark');
};

// @route POST /api/quiz/submit
// @desc Submits quiz answers, calculates result, and returns product recommendations
// @access Public 
router.post('/submit', async (req, res) => {
    // Destructuring with defensive defaults
    const { email, answers = {}, name } = req.body; 

    // Basic Validation: answers is now guaranteed to be an object due to default value
    if (!name || Object.keys(answers).length < 1) {
        return res.status(400).json({ message: 'Name and complete answers are required for quiz submission.' });
    }

    try {
        // 1. Calculate Result
        const scentType = calculateScentType(answers);
        const recommendationNames = getRecommendations(scentType);

        // 2. Find full product data from DB
        const productRecommendations = await Product.find({ name: { $in: recommendationNames } });

        // 3. Format recommendations for frontend/email
        const formattedRecommendations = productRecommendations.map(p => ({
            name: p.name,
            price: `€${p.price.toFixed(2)}`,
            description: p.moodDescription || p.description || 'A unique fragrance.',
            imageUrl: p.productImageUrl || 'https://placehold.co/400x500/f0f0f0/333?text=Miragem',
            link: `/fragrances/${p.name.toLowerCase().replace(/ /g, '-')}`
        }));

        // 4. Record Quiz Submission (optional: add this back if QuizSubmission model is implemented)
    
        await QuizSubmission.create({
            userName: name,
            email: email || undefined,
            answers: answers,
            scentType: scentType,
            // ... map product IDs here
        });
     

        // 5. Update Waitlist & Send Email
        if (email) {
            await Waitlist.findOneAndUpdate(
                { email },
                { $set: { quizResult: scentType, recommendations: recommendationNames } },
                { new: true, upsert: true } 
            );
            
            // Send email with recommendations (assuming sendQuizRecommendationEmail is available)
             await sendQuizRecommendationEmail(email, name, scentType, formattedRecommendations);
        }

        res.status(200).json({ 
            scentType: scentType, 
            recommendations: formattedRecommendations,
            message: 'Quiz result processed successfully.'
        });

    } catch (error) {
        console.error('Error processing quiz submission:', error);
        res.status(500).json({ message: 'Server error processing quiz results.' });
    }
});

// @route GET /api/quiz/questions
// @desc Get all active quiz questions
// @access Public 
router.get('/questions', async (req, res) => {
    try {
        const QuizQuestion = require('../models/QuizQuestion');
        const questions = await QuizQuestion.find().sort({ order: 1 });

        // If DB is empty, return a small mock set for quick testing
        if (questions.length > 0) {
            return res.status(200).json(questions);
        }

        const mockQuestions = [
            { _id: '60c72b2f9c1d440000000001', order: 1, question: "What is your name?", stepType: 'input', isMandatory: true, options: [{ id: 'name', text: 'First Name', value: 'name' }] },
            { _id: '60c72b2f9c1d440000000002', order: 2, question: "I am buying for", stepType: 'choice', isMandatory: true, options: [{ id: 'myself', text: 'Myself', value: 'self' }, { id: 'someone_else', text: 'Someone Else', value: 'gift' }] },
            { _id: '60c72b2f9c1d440000000003', order: 3, question: "The colour palette I feel reflects me best is", stepType: 'image-choice', isMandatory: true, options: [{ id: 'soft', text: 'Soft', value: 'floral', image: 'https://placehold.co/400x400/F0CC6E/182A2E?text=Soft' }, { id: 'rich', text: 'Rich', value: 'warm', image: 'https://placehold.co/400x400/182A2E/F0CC6E?text=Rich' }, { id: 'bright', text: 'Bright', value: 'fresh', image: 'https://placehold.co/400x400/FAF7F0/182A2E?text=Bright' }, { id: 'dark', text: 'Dark', value: 'dark', image: 'https://placehold.co/400x400/000000/FAF7F0?text=Dark' }] },
        ];
        res.status(200).json(mockQuestions);

    } catch (error) {
        console.error('Error fetching quiz questions:', error);
        res.status(500).json({ message: 'Server error fetching quiz questions.' });
    }
});

module.exports = router;